/**
 * \file sdc_random.h
 *
 * \brief libSDC: Definition of random generator functions
 *
 * Applications which want to use the random generator provided by libSDC
 * need to include this header file
 *
 * \author Christoph Gellner (cgellner@de.adit-jv.com)
 *
 * \copyright (c) 2015 Advanced Driver Information Technology.
 * This code is developed by Advanced Driver Information Technology.
 * Copyright of Advanced Driver Information Technology, Bosch, and DENSO.
 * All rights reserved.
 *
 *
 ***********************************************************************/

/**
 * \ingroup libSDC_API
 * \defgroup random Random Generator
 * Functions to manage and use libSDC Random generator
 */

#ifndef __SDC_LIB_RANDOM_H_
#define __SDC_LIB_RANDOM_H_

#ifdef __cplusplus
extern "C"
{
#endif

#include <sdc.h>
#include <sdc_session.h>

/* Definitions types and defaults */

/* Functions */

/**
 * \ingroup random
 * \brief Generate a buffer of random data
 *
 * The function will internally allocate a buffer of the desired length and
 * fill it with random data.
 * On success this buffer is returned using \p random_data pointer.
 * In this case the buffer needs to be freed externally.
 * \todo Add support for HW random generator - currently /dev/urandom is used internally
 *
 * \param[in] session       session handle created using ::sdc_open_session
 * \param[in] len           desired length
 * \param[out] random_data  pointer to store the allocated buffer to
 *
 * \return \ref SDC_OK - no error
 * \return \ref SDC_SESSION_INVALID - given session is invalid
 * \return \ref SDC_INVALID_PARAMETER - invalid buffer or length
 * \return \ref SDC_NO_MEM - failed to allocate random buffer
 * \return otherwise an unexpected error occurred that should be handled by default
 */
SDC_API sdc_error_t sdc_random_gen_buffer (sdc_session_t *session, const size_t len, uint8_t **random_data);

/**
 * \ingroup random
 * \brief Generate random data
 *
 * The function will fill the specified buffer with random data.
 *
 * \param session       session handle created using ::sdc_open_session
 * \param len           desired length
 * \param random_data   pointer to store the allocated buffer to
 *
 * \return \ref SDC_OK - no error
 * \return \ref SDC_SESSION_INVALID - given session is invalid
 * \return Can return an architecture specific error code as well.
 */
SDC_API sdc_error_t sdc_random_gen(sdc_session_t *session, const size_t len,
                                   uint8_t *random_data);

#ifdef __cplusplus
}
#endif

#endif
